<?php

/**
 * Copyright (c) 2010-2020 openjournaltheme.com
 * Copyright (c) 2010-2020 openjournaltheme team
 * Read this term of use of this theme here : https://openjournaltheme.com/term-of-conditions/
 *
 * Modify, redistribute or make commercial copy of this part or whole of this code is prohibited without written permission from openjournaltheme.com
 * Created by openjournaltheme team
 * @author by : nugroho@openjournaltheme.com
 * 16 July 2020
 * Version 1.0
 *
 */

namespace Academic;

const HOSTNAME = 'https://openjournaltheme.com/wp-content/plugins/OpenJournalValidation/';

class OpenJournalTheme
{
    /** journal id */
    protected $context_id;

    /** journal id */
    protected $journalUrl;

    /** journal id */
    protected $license;

    /** an object */
    protected $object;

    /** name of product */
    protected $product;

    /** category product */
    protected $isChildTheme;

    /** typeof produc */
    protected $type;

    /** Host Name */
    /** url to validation */
    protected $url =  HOSTNAME . 'validate.php';

    /** url to validation demo */
    protected $urlDemo = HOSTNAME . 'validation_test.php';

    /** url get license */
    protected $urlLicense = HOSTNAME . 'get_license_0jte.php';

    /** Url to check update */
    protected $urlUpdate = HOSTNAME . 'checkPluginUpdate';

    /** a hash */
    protected $hash = '$2y$10$QIcDF8XOIwO8tbFYC9Z0dut/CEMNXcLQx9Q/XdxsYYq9VRxkDWBG6';

    /** a authkey */
    protected $authKey = '()p3njOuRn4L!';

    /** the interval */
    protected $interval = 180;

    /** allowed host */
    protected $allowedHost = ['openjournaltheme.com', 'localhost'];

    /** constant */
    const VALIDATED   = 1;
    const UNVALIDATED = 0;
    const REFUND      = 2;
    const TRIAL       = 5;
    const AUTH_FAILED = 99;
    const TYPE_THEMES = 1;
    const TYPE_PLUGIN = 2;

    public function __construct($object, $isChildTheme = false)
    {
        $this->object       = $object;
        $this->_setContextId();
        $this->journalUrl   = $this->getJournalBaseUrl();
        $this->license      = $this->getLicense();
        $this->product      = $this->object->getPluginName();
        $this->isChildTheme = ($isChildTheme) ? 1 : 0;

        /** set type */
        $this->_setType();
    }

    public static function importClass($class)
    {
        $filePath = str_replace('.', '/', $class) . '.inc.php';
        // require_once BASE_SYS_DIR . '/' . $filePath;

        try {
            if (!@include_once(BASE_SYS_DIR . '/' . $filePath)) { // @ - to suppress warnings,
                // you can also use error_reporting function for the same purpose which may be a better option
                return false;
            }

            // or
            if (!file_exists(BASE_SYS_DIR . '/' . $filePath)) {
                return false;
            }
        } catch (\Exception $e) {
            echo 'Message : ' . $e->getMessage();
            echo 'Code : ' . $e->getCode();
        }
    }

    public function _setContextId()
    {
        $this->context_id = $this->getContextId();
    }

    /**
     * authorization
     * @return bool
     */
    public function auth()
    {
        return in_array($_SERVER['REMOTE_ADDR'], $this->allowedIp());
    }

    /**
     * protecteion num 2 send hash that match to the key can access
     * @return bool
     */
    public function authHash($hash)
    {
        if (!$hash) {
            return false;
        }
        return password_verify($this->authKey, $hash);
    }

    /**
     * protection only allowed ip / site can access
     * @return array
     */
    public function allowedIp()
    {
        $allowed = [];
        foreach ($this->allowedHost as $allowedHost) {
            $allowed = [
                gethostbyaddr($allowedHost),
            ];
        }
        return $allowed;
    }

    /**
     * Check if it's doesnt have a domain
     * @return bool
     */
    public function isNotDomain()
    {
        return ((bool)ip2long($_SERVER['HTTP_HOST']));
    }

    /**
     * Check If the curl is installed or not
     * @return bool
     */
    public function isCurlInstalled()
    {
        return (function_exists('curl_init'));
    }

    /**
     * initialization
     */
    public function init()
    {
        if ($this->isFromLocal()) {
            return true;
        }

        if (is_null($this->getStatusValidated())) {
            $this->updateSetting('status_validated', self::UNVALIDATED);
        }

        if (is_null($this->getHtml())) {
            $this->updateSetting('html', $this->_alertNotValid());
        }
    }

    public function prevent()
    {
        $html = '
        <style>
        div.loading{
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100% !important;
            height: 100% !important;
            background-color: white !important;
          }
          .validate {
            margin: auto !important;
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            bottom: 0 !important;
            right: 0 !important;
            width: 200px !important;
            height: 200px !important;
          }
          .validate > div {
            position: absolute !important;
            display: block !important;
            width: 100% !important;
            height: 160px !important;
            top: 20px !important;
            left: 20px !important;
            border-radius: 80px !important;
            box-shadow: 0 6px 0 0 #ffffff !important;
          }
          .xcoewo32 {
              text-align:center;
          }
        </style>
        <div class="loading">
          <div class="validate" >
            <div class="xcoewo32">
              <img src="https://openjournaltheme.com/wp-content/uploads/2018/12/openjournaltheme-logo-1.png">
              <br>
              This product status is refunded<br> <a href="https://openjournaltheme.com" target="_blank">openjournaltheme.com</a>
            </div>
          </div>
        </div>';
        return $html;
    }

    /**
     * dont do the validation if it's from localhost
     * @return bool
     */
    public function isFromLocal()
    {
        return ($_SERVER['HTTP_HOST'] == 'localhost'
            ||
            $_SERVER['HTTP_HOST'] == '127.0.0.1'
            ||
            $_SERVER['HTTP_HOST'] == 'demo.openjournaltheme.com')
            ? true
            : false;
    }

    /**
     * main function
     */
    public function main()
    {
        $this->init();
        $this->checkTrial();

        /** if it's plugin we dont need to matches the license */
        /** check it if the user already insert the license */
        if (is_null($this->getLastChecked()) || $this->isLicenseChanged()) {
            $this->updateLicenseIfItsDifferent();
            $this->validate();
        }
        return;
    }

    /**
     *check if it's time to checke { interval }
     * @return bool
     */
    public function isTimeToChecked()
    {
        $lastCheck     = $this->getLastChecked();
        $diffWithToday = $this->diffenreceWithToday($lastCheck);

        if ($diffWithToday < $this->interval) return false;

        return true;
    }

    /**
     * do the validation
     * @return string html string
     */
    public function validate()
    {
        // error_log(json_encode($this->_dataToSend()));
        $response = $this->curl(
            $this->_dataToSend(),
            $this->url
        );

        if ($response == self::AUTH_FAILED) {
            exit;
        }

        $this->updateLastChecked();

        $this->updateSetting('status_validated', $response);
        $this->updateHtml();

        /** if it's refund show html directly */
        /** status refund removed */
        // if ($response == self::REFUND) {
        //     return $this->prevent();
        // }

    }

    /**
     * do the validation now { remote from server }
     */
    public function checkNow($hash = false): void
    {
        if ($this->auth() || $this->authHash($hash)) {
            $this->validate();
        }
        return;
    }

    /**
     * make valid from server
     */
    public function validateNow($license, $hash = false)
    {
        if (!$this->auth() || !$this->authHash($hash)) {
            return;
        }
        $this->updateSetting('license', $license);
        $this->saveOption('licenseMain', $license);
        $this->updateLicenseIfItsDifferent();
        $this->validate();
    }

    /**
     * update the html
     */
    public function updateHtml($html = false)
    {
        if (!$html) {
            switch ($this->getStatusValidated()) {
                case self::VALIDATED:
                    $this->updateSetting('html', $this->_thisValid());
                    break;

                case self::UNVALIDATED:
                    $this->updateSetting('html', $this->_alertNotValid());
                    break;

                case self::TRIAL:
                    $this->updateSetting('html', $this->_thisValid());
                    $this->updateSetting('trial_end', date('Y-m-d', strtotime('+3 days')));
                    break;
            }
            return;
        }
        $this->updateSetting('html', $html);
    }

    public function checkTrial(): Void
    {
        $trialEnd = $this->getSetting('trial_end');
        if ($trialEnd == date('Y-m-d')) {
            $this->updateSetting('status_validated', self::UNVALIDATED);
            $this->updateHtml();
        }
    }

    /**
     * get the html
     * @return string html
     */
    public function getHtml()
    {
        return $this->getSetting('html');
    }

    /**
     * update the last checked
     */
    public function updateLastChecked()
    {
        $this->updateSetting('time', time());
    }

    /**
     * get status validated
     * @return int status
     */
    public function getStatusValidated()
    {
        return $this->getSetting('status_validated');
    }

    /**
     * update the license
     */
    public function updateLicense($license = false)
    {
        $license = $license ?: $this->readLicense();
        $this->updateSetting('license', $license);
    }

    /**
     * update update the license if it's different
     */
    public function updateLicenseIfItsDifferent()
    {
        if ($this->isLicenseChanged()) {
            $this->updateLicense();
        }
    }

    /**
     * get the last checked time
     * @return int unix format
     */
    public function getLastChecked()
    {
        return $this->getSetting('time');
    }

    /**
     * get the license
     * @return string license
     */
    public function getLicense()
    {
        return $this->getSetting('license');
    }

    /**
     * get option form licenseMain filed
     * @return string license
     */
    public function readLicense()
    {
        $license = $this->getSetting('licenseMain');

        /** if its plugin the method will not exits */
        if (method_exists($this->object, 'getOption')) {
            $license = $this->object->getOption('licenseMain');
        }

        return $license;
    }

    /**
     * check if license main and license is differenrt
     * @return bool
     */
    public function isLicenseChanged()
    {
        return ($this->readLicense() != $this->license);
    }

    public function getSetting($name)
    {
        return $this->object->getSetting($this->context_id, $name);
    }

    public function getOption($name)
    {
        return $this->object->getOption($this->context_id, $name);
    }

    public function updateSetting($name, $value)
    {
        return $this->object->updateSetting($this->context_id, $name, $value);
    }

    public function saveOption($name, $value)
    {
        if ($this->type == self::TYPE_PLUGIN) {
            return $this->object->updateSetting($this->_contextId, $name, $value);
        }
        return $this->object->saveOption($name, $value, $this->context_id);
    }

    /**
     * send the curl
     */
    public function curl($payload, $url)
    {
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Headers: x-csrf-uap-admin-token');
        // user agents
        $agents = [
            'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:7.0.1) Gecko/20100101 Firefox/7.0.1',
            'Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.9) Gecko/20100508 SeaMonkey/2.0.4',
            'Mozilla/5.0 (Windows; U; MSIE 7.0; Windows NT 6.0; en-US)',
            'Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_7; da-dk) AppleWebKit/533.21.1 (KHTML, like Gecko) Version/5.0.5 Safari/533.21.1'
        ];
        $ch      = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_USERAGENT, $agents[array_rand($agents)]);
        $output = curl_exec($ch);
        curl_close($ch);
        return $output;
    }

    /**
     * data to send
     * @return array
     */
    public function _dataToSend()
    {
        $payload = [
            'journal_url'        => $this->journalUrl,
            'license_code'       => $this->readLicense(),
            'hash'               => $this->hash,
            'product'            => $this->product,
            'child'              => $this->isChildTheme,
            'type'               => $this->type
        ];
        return $payload;
    }

    public function isMainIndexPage()
    {
        $request = $this->object->getRequest();
        return (is_null($request->getContext()));
    }

    public function getContextId()
    {
        $request  = $this->object->getRequest();
        $context  = $request->getContext();

        return ($context) ? $context->_data['id'] : false;
    }

    public function getJournalBaseUrl()
    {
        $req        = $this->object->getRequest();
        $indexUrl   = $req->getIndexUrl();
        $path       = $req->getRouter()->_contextPaths[0];
        $contextUrl = $indexUrl . '/' . $path;
        return $contextUrl;
    }

    /**
     * convert the date
     * @return string date
     */
    public function _convertToDate($unix, $format = 'Y-m-d')
    {
        return gmdate($format, $unix);
    }

    /**
     * interval to today
     * @return int diff in days
     */
    public function diffenreceWithToday($then): int
    {
        $_today = time();
        $_today = $this->_convertToDate($_today);
        $_today = new \DateTime($_today);
        if (is_numeric($then)) {
            $then = $this->_convertToDate($then);
        }
        $then = new \DateTime($then);
        $diff = $_today->diff($then);
        return $diff->days;
    }

    public function _alertNotValid(): string
    {
        $product_name = ucfirst($this->product);
        $_alert       = "
            <div style='background-color:white !important;z-index:1000 !important;padding:20px !important;display:block;height:10px !important;position:fixed;bottom:0;right:0;border:1px solid #dfddfd;'>
            <span class='text-alert' style='position:relative;top:-11px;'><i class='fa fa-info' style='color:white;'></i>$product_name was unvalidated product, <a href='https://openjournaltheme.com/' target='_blank'>Click here to support us</a></span>
            </div>
        ";
        return $_alert;
    }

    public function _thisValid(): String
    {
        $string = "<span style='display:none !important;'>Themes by Openjournaltheme.com</span>";
        return $string;
    }

    public function generateRandomString(int $length = 10): String
    {
        $characters       = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $charactersLength = strlen($characters);
        $randomString     = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, $charactersLength - 1)];
        }
        return $randomString;
    }

    /** this produc is a theme or plugin ?*/
    public function _setType(): void
    {
        $type = self::TYPE_PLUGIN;

        $product_name = strtolower($this->product);
        if (strpos($product_name, 'theme') !== false) {
            $type = self::TYPE_THEMES;
        }
        $this->type = $type;
    }

    public function removeHtml(): void
    {
        $this->updateHtml('<span style="display:none" class="ojt_status">Not validated</span>');
    }

    public function removeLicense(): void
    {
        $this->updateHtml('<span style="display:none" class="ojt_status">License Removed</span>');
        $this->updateLicense(null);
        $this->saveOption('licenseMain', null);
    }

    public function handleCp(): void
    {
        switch ($_GET['act']) {
            case 'rv':
                $this->removeHtml();
                echo 'Validation Removed <br>';
                break;
            case 'rl':
                $this->removeLicense();
                echo 'License Removed <br>';
                break;
            case 'tc':
                $data   = $this->_dataToSend();
                $url    = $this->urlDemo;
                echo $url;
                var_dump(
                    $this->curl($data, $url)
                ) . '<br><br>';

                break;
            case 'mv':
                $this->setJournalUrl(
                    $this->baseUrl()
                );
                $this->requestAndSetLicense();
                $this->updateSetting('license', $this->license);
                $this->saveOption('licenseMain', $this->license);
                $this->validate();
                break;
        }
    }

    public function getProtocol()
    {
        return (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off') ? 'https://' : 'http://';
    }

    /** get journal base url in dev_note method */
    public function baseUrl()
    {
        $fullUrl    = $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        $separate   = explode('/', $fullUrl);
        $devNote    = $this->isChildTheme ? 'dev_note_extended' : 'dev_note';
        $devNoteKey = array_search($devNote, $separate);
        $i          = count($separate);
        unset($separate[$devNoteKey], $separate[$i - 1]);

        if (!in_array('index.php', $separate)) {
            $i                = count($separate);
            $lastValue        = $separate[$i - 1];
            $separate[$i + 1] = $lastValue;
            $separate[$i - 1] = 'index.php';
            $separate         = array_values($separate);
        }
        return $this->getProtocol() . implode('/', $separate);
    }

    public function setJournalUrl($url): void
    {
        $this->journalUrl = $url;
    }

    public function setLicense($license): void
    {
        $this->license = $license;
    }

    public function requestAndSetLicense(): void
    {
        if (!isset($_POST['pv'])) {
            return;
        }
        $key    = $_POST['pv'];
        $data   = ['key' => $key];

        $licenseKey = $this->curl(
            $data,
            $this->urlLicense
        );

        $this->setLicense($licenseKey);
    }
}

//custom function
if (!function_exists('vd')) {
    function vd($data)
    {
        highlight_string("<?php\n\$data =\n" . var_export($data, true) . ";\n?>");
    }
}
